/*
 * Copyright (C) 1997-2003 by Objective Systems, Inc.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by Objective Systems, Inc.
 *
 * PROPRIETARY NOTICE
 *
 * This software is an unpublished work subject to a confidentiality agreement
 * and is protected by copyright and trade secret law.  Unauthorized copying,
 * redistribution or other use of this work is prohibited.
 *
 * The above notice of copyright on this source code product does not indicate
 * any actual or intended publication of such source code.
 *
 *****************************************************************************/

/** 
 * @file OSCInputStream.h 
 * Input stream class definitions
 */
#ifndef _OSCINPUTSTREAM_H_
#define _OSCINPUTSTREAM_H_

#include "OSCStream.h"
#include "OSCSocket.h"

#if !defined NO_ASN1_RT_REDUNDANT
/** @defgroup istrmclas Generic Input Stream Classes
 *  @ingroup cppruntime
 * These classes are used to input data from the various stream types. 
 * They are not used for ASN.1 decoding. The streams managed by these 
 * classes are unbuffered (i.e. all I/O is done directly to the device, 
 * with no intermediate memory buffer). 
 * @{
 */

/**
 * Abstract generic input stream base class.
 */
class EXTERNRT OSCInputStream : public OSCStream {
 protected:
   /**
    * The default constructor.  It is protected indicating that this 
    * is an abstract base class.
    *
    * @exception OSCStreamException    stream can't be created or initialized.
    * @see                            ::rtStreamInit
    */
   OSCInputStream ();

 public:
   /**
    * The destructor. Closes the stream if necessary.
    */
   virtual ~OSCInputStream ();

   /**
    * This method returns the current position in the stream (in octets).
    *
    * @return The number of octets already read from the stream.
    */
   virtual ASN1UINT currentPos();

   /**
    * Read data from the stream.  This method reads up to \c maxToRead 
    * bytes from the stream.  It may return a value less then this if 
    * the mamimum number of bytes is not available.
    *
    * @param destBuf                   Pointer to a buffer to receive a data.
    * @param maxToRead                 Size of the buffer.
    * @return                          The total number of octets read into the
    *                                    buffer.
    * @exception OSCStreamException    data can't be read from the stream.
    * @see                            ::rtStreamRead
    */
   virtual ASN1UINT read (ASN1OCTET* destBuf, ASN1UINT maxToRead);
} ;

/**
 * Generic file input stream.  This class opens an existing file for 
 * input in binary mode and reads data from it.
 */
class EXTERNRT OSCFileInputStream : public OSCInputStream {
 public:
   /**
    * Creates and initializes a file input stream using the name of file.
    *
    * @param pFilename                 Name of file.
    * @exception OSCStreamException    stream can't be created or initialized.
    * @see                            ::rtStreamFileOpen
    */
   OSCFileInputStream (const char* pFilename);

   /**
    * Initializes the file input stream using the opened FILE structure
    * descriptor.
    *
    * @param file                      Pointer to FILE structure.
    * @exception OSCStreamException    stream can't be created or initialized.
    * @see                            ::rtStreamFileAttach
    */
   OSCFileInputStream (FILE* file);
} ;

/**
 * Generic socket input stream.  This class is used to read data 
 * from a TCP/IP socket.
 */
class EXTERNRT OSCSocketInputStream : public OSCInputStream {
 protected:
   OSCSocket mSocket; ///< a socket 
 public:
   /**
    * Creates and initializes the socket input stream using the socket's
    * handle.
    *
    * @param socket                    Handle of the socket.
    * @param ownership                 Indicates ownership of the socket. 
    *                                  Set to TRUE to pass ownership to 
    *                                  this object instance.  The socket 
    *                                  will be closed when this object 
    *                                  instance is deleted or goes out 
    *                                  of scope.
    * @exception OSCStreamException    stream can't be created or initialized.
    * @see                            ::rtSocketCreate, ::rtSocketAccept
    * @see                            ::rtStreamSocketAttach
    */
   OSCSocketInputStream (OSRTSOCKET socket, ASN1BOOL ownership = FALSE);

   /**
    * Initializes socket input stream using an instance of an OSCSocket class.
    *
    * @param socket                    Reference to OSCSocket instance.
    * @exception OSCStreamException    stream can't be created or initialized.
    */
   OSCSocketInputStream (OSCSocket& socket);
} ;

/**
 * Generic memory input stream.  This class is used to read data from 
 * a memory buffer (i.e. an array of bytes in memory).
 */
class EXTERNRT OSCMemoryInputStream : public OSCInputStream {
 public:
   /**
    * Initializes the memory input stream using the specified memory buffer.
    *
    * @param pMemBuf                   The pointer to the buffer.
    * @param bufSize                   The size of the buffer.
    * @exception OSCStreamException    stream can't be created or initialized.
    * @see                            ::rtStreamMemoryAttach
    */
   OSCMemoryInputStream (ASN1OCTET* pMemBuf, ASN1UINT bufSize);
} ;

/** @} */

#endif	/* !NO_ASN1_RT_REDUNDANT */

#endif // _OSCINPUTSTREAM_H_
