/* vim:set sw=4 ts=8 fileencoding=cp1251:::WINDOWS-1251[] */
/*
 * Copyright(C) 2005-2013  
 *
 *    , 
 *    .
 *
 *        ,
 * ,    ,
 *     ,
 * ,      
 *     
 *      .
 *
 *  ,    , 
 *         
 *   .
 *
 *  -   
 *     .
 */

/*
 * \version $Revision: 127051 $
 * \date $Date:: 2015-09-09 15:08:20 +0300#$
 * \author $Author: pav $
 */

#define CERT_REVOCATION_PARA_HAS_EXTRA_FIELDS 
#ifdef HAVE_CPRO_CONFIG_H
#include "myconfig.h"
#endif
#ifdef _WIN32
#   include <Windows.h>
#   include <WinCrypt.h>
#endif
#include "wincspc.h"
#include <vector>
#include <string.h>
#include <math.h>

#include "typedefs.h"
#include "ike_gost.h"
#include "WinCryptEx.h"

#include "ike_cert.h"
#include "example.h"

#ifdef TEST_CONFORMITY
#   include "../../Conformity/ConformityIPSEC.h"
#endif  /* TEST_CONFORMITY */

// Convert 100-ns interval since January 1, 1601 (UTC)
// to 1-sec interval science January 1, 1970, UTC
/*!
* \ingroup internalIKE
* \brief   FILETIME  time_t
*
*  100-   January 1, 1601 (UTC)
*  1-   January 1, 1970, UTC
*
* \param lpFT [out]   FILETIME.
*
* \return   
*/
static time_t 
filetime2time_t(const FILETIME* lpFT)
{
    long double ldtime;
    time_t time;
    ldtime = ((ldexpl(lpFT->dwHighDateTime, 32) + 
	       lpFT->dwLowDateTime
	      )*100.e-9
	     ) - 11644473600.;
	//SystemTimeToFileTime({.wYear = 1970, .wMonth = 1, .wDay = 1}...
    time = (time_t)ldtime;
    if(0. >= ldtime ||
       (ldtime - 1.) > time) {
	return (time_t)(-1); // See C99 "7.23.2.3 The mktime function"
    }
    return time;
}

/*!
 * \ingroup internalIKE
 * \brief   
 *
 * \param h [in]  ;
 * \param szOID [in]    (  CryptDecodeObject).
 * \param pbData [in]     .
 * \param cbData [in]      .
 * \param decodedValue [out]   .      printf().
 *
 * \return true  
 * \return false 
 */
static bool
ikeDecodeExtension(ike_gost_handle h,
		   LPCSTR szOID, BYTE *pbData, DWORD cbData, 
		   std::vector<BYTE>& decodedValue)
{
    DWORD dwSize = 0;
    if(!CryptDecodeObject(
	X509_ASN_ENCODING | PKCS_7_ASN_ENCODING,
	szOID,
	pbData,
	cbData,
	0,
	0,
	&dwSize)) {
	    h->LogUfn(h->pUfnArg, CAPI_ERR, 0, "Err: %#x.", GetLastError());
	    return false;
    }
    decodedValue.resize(dwSize);
    if(!CryptDecodeObject(
	X509_ASN_ENCODING | PKCS_7_ASN_ENCODING,
	szOID,
	pbData,
	cbData,
	0,
	&decodedValue[0],
	&dwSize)) {
	    h->LogUfn(h->pUfnArg, CAPI_ERR, 0, "Err: %#x.", GetLastError());
	    return false;
    }

    return true;
}

/*!
 * \ingroup ikeexample
 * \brief     
 *
 * \xmlonly <locfile><header>ike_cert.h</header></locfile>\endxmlonly
 *
 *       
 *     ike_cert.cpp.
 *
 *   privateKeyUsagePeriod  ,    
 *       .
 *
 *   privateKeyUsagePeriod.NotBefore  ,   
 *         
 * .
 * 
 *  privateKeyUsagePeriod.NotBefore    
 * ,    .
 *
 * \param h [in]  ;
 * \param pCertContext [in]   
 * \param pftNotAfter [in]     
 *  
 * \return true  
 * \return false 
 */
static bool
ikeCheckPrivateKeyUsagePeriod(ike_gost_handle h, 
			      PCCERT_CONTEXT pCertContext, 
			      FILETIME *pftNotAfter)
{
    FILETIME notAfter = pCertContext->pCertInfo->NotAfter;
    FILETIME notBefore = pCertContext->pCertInfo->NotBefore;

    memset(pftNotAfter, 0, sizeof(*pftNotAfter));

    PCERT_EXTENSION pPkup = ::CertFindExtension(
        szOID_PRIVATEKEY_USAGE_PERIOD,
        pCertContext->pCertInfo->cExtension,
        pCertContext->pCertInfo->rgExtension);
    if(pPkup) {
	std::vector<BYTE> decodedPkupValue;
	if(!ikeDecodeExtension(h, szCPGUID_PRIVATEKEY_USAGE_PERIOD_Encode,
			       pPkup->Value.pbData, pPkup->Value.cbData, decodedPkupValue)) {
	    return false;
	}
	PCPCERT_PRIVATEKEY_USAGE_PERIOD pPkupValue = 
	    reinterpret_cast<PCPCERT_PRIVATEKEY_USAGE_PERIOD>(&decodedPkupValue[0]);

	if( !pPkupValue->pNotAfter ) {
	    h->LogUfn(h->pUfnArg, CAPI_ERR, 0, 
		    "PKUP extension has no notAfter value", GetLastError());
	    return false;
	}
	notAfter = *pPkupValue->pNotAfter;
	if( pPkupValue->pNotBefore ) {
	    notBefore = *pPkupValue->pNotBefore;
	}
    }

    FILETIME now;
    SYSTEMTIME stNow;
    ::GetSystemTime(&stNow);
    if(!::SystemTimeToFileTime(&stNow,&now)) {
        h->LogUfn(h->pUfnArg, CAPI_ERR, 0, 
		"SystemTime to FileTime conversion failed", GetLastError());
        return false;
    }

    if( 0 > CompareFileTime(&now,&notBefore)
        || 0 < CompareFileTime(&now,&notAfter) ) {
        h->LogUfn(h->pUfnArg, CAPI_ERR, 0, 
		"Private key cannot be used at the time", GetLastError());
        return false;
    }

    *pftNotAfter = notAfter;
    return true;
}

/*!
 * \ingroup ikeexample
 * \brief    .
 *
 * \xmlonly <locfile><header>ike_cert.h</header></locfile>\endxmlonly
 *
 *       
 *     ike_cert.cpp.
 *
 * \param h [in]  ;
 * \param pbCert [in]  
 * \param uCertLen [in]   
 * \param ppCertCtx [out]  ,    
 * \param ptNotAfter [out]      
 *			     (CRL).
 *
 * \return CAPI_NOERROR  
 * \return CAPI_CALL_ERROR   
 * \return CAPI_CHAIN_ERROR       
 *
 * \par SEE ALSO
 * #p1_SetParamFn (), #p2_SetParamFn ()
 */
capi_result
ikeCheckCert(ike_gost_handle h, const BYTE* pbCert, unsigned uCertLen, 
	     PCCERT_CONTEXT *ppCertCtx, time_t *ptNotAfter)
{
    if(!h)
	return CAPI_CALL_ERROR;

    if ( !ppCertCtx ) {
	h->LogUfn(h->pUfnArg, CAPI_ERR, 0, "NULL args");
	return CAPI_CHAIN_ERROR;
    }
    PCCERT_CONTEXT pCertContext;
    if( !pbCert )
    {
	pCertContext = *ppCertCtx;
	if ( !pCertContext ) {
	    h->LogUfn(h->pUfnArg, CAPI_ERR, 0, "NULL args");
	    return CAPI_CHAIN_ERROR;
	}
    }
    else
    {
	// Open context
	pCertContext = ::CertCreateCertificateContext(
	    X509_ASN_ENCODING | PKCS_7_ASN_ENCODING,
	    pbCert, uCertLen);
	if(!pCertContext) {
	    h->LogUfn(h->pUfnArg, CAPI_ERR, 0, "Err: %#x.", GetLastError());
	    return CAPI_CHAIN_ERROR;
	}   
    }
    *ptNotAfter = 0;

    // Check private key usage period
    FILETIME ftNotAfter;

    if(!ikeCheckPrivateKeyUsagePeriod(h, pCertContext, &ftNotAfter)) {
	h->LogUfn(h->pUfnArg, CAPI_ERR, 0, "ikeCheckPrivateKeyUsagePeriod failed");
	::CertFreeCertificateContext(pCertContext);
	return CAPI_CHAIN_ERROR;
    }

    // Build chain and check for revocation (except root)
    CERT_ENHKEY_USAGE    EnhkeyUsage;
    CERT_USAGE_MATCH     CertUsage;
    CERT_CHAIN_PARA      ChainPara;
    static LPCSTR	 usagesIDs[] = {
	// RFC 2459,  
		//"1.3.6.1.5.5.7.3.5",        // id-kp-clientAuth 
		//"1.3.6.1.5.5.7.3.6",        // id-kp-ipsecTunnel
		//"1.3.6.1.5.5.7.3.7",        // id-kp-ipsecUser
	// Microsoft, ,    draft-ietf-ipsec-pki-req
		"1.3.6.1.5.5.8.2.2",	    // IP security IKE intermediate
	// RFC 4945,  MS ,    id-kp-ipsecIKE
		"1.3.6.1.5.5.7.3.17",       // id-kp-ipsecIKE
	    };

    EnhkeyUsage.cUsageIdentifier = sizeof(usagesIDs)/sizeof(*usagesIDs);
    EnhkeyUsage.rgpszUsageIdentifier = (LPSTR*)usagesIDs;
    CertUsage.dwType = USAGE_MATCH_TYPE_OR;
    CertUsage.Usage  = EnhkeyUsage;
    ChainPara.cbSize = sizeof(CERT_CHAIN_PARA);
    ChainPara.RequestedUsage=CertUsage;
#ifndef TEST_CONFORMITY
    PCCERT_CHAIN_CONTEXT pChainContext;
    if(!::CertGetCertificateChain(
	   0,
	   pCertContext,
	   0,
	   0,
	   &ChainPara,
	   CERT_CHAIN_REVOCATION_CHECK_CHAIN |	//     
	   CERT_CHAIN_CACHE_END_CERT,		// ,  
	   0,
	   &pChainContext)) {
        h->LogUfn(h->pUfnArg, CAPI_ERR, 0, "Err: %#x.", GetLastError());
        ::CertFreeCertificateContext(pCertContext);
        return CAPI_CHAIN_ERROR;
    }

    // Check chain status
    if(pChainContext->TrustStatus.dwErrorStatus != CERT_TRUST_NO_ERROR) {
        h->LogUfn(h->pUfnArg, CAPI_ERR, 0, "Chain dwErrorStatus: %#x.", 
            pChainContext->TrustStatus.dwErrorStatus);
        ::CertFreeCertificateChain(pChainContext);
        ::CertFreeCertificateContext(pCertContext);
        return CAPI_CHAIN_ERROR;
    }

    CERT_CHAIN_POLICY_PARA policyPara = { sizeof(CERT_CHAIN_POLICY_PARA) };
    CERT_CHAIN_POLICY_STATUS policyStatus = { sizeof(CERT_CHAIN_POLICY_STATUS) };
    if(!::CertVerifyCertificateChainPolicy(CERT_CHAIN_POLICY_BASE,
					   pChainContext, &policyPara, &policyStatus)) {
        h->LogUfn(h->pUfnArg, CAPI_ERR, 0,
            "Certificate chain policy check failed, policyStatus.dwError = %d", policyStatus.dwError);
        ::CertFreeCertificateChain(pChainContext);
        ::CertFreeCertificateContext(pCertContext);
        return CAPI_CHAIN_ERROR;
    }

    // RFC 4301 Section 4.4.2.1 of [ARCH] 
    // <http://tools.ietf.org/html/rfc4301#section-4.4.2.1>
    // "...If time is employed, and if
    // IKE employs X.509 certificates for SA establishment, the SA
    // lifetime must be constrained by the validity intervals of the
    // certificates, and the NextIssueDate of the Certificate Revocation
    // Lists (CRLs) used in the IKE exchange for the SA..."
    //
    //  (lse)  -    
    // (certificates, CRLs) [ARCH]     
    // , ,  ,       
    //  .   NextIssueDate,  ,  
    //  ,  NextUpdate.
    //
    //  ,    
    // ( pChainContext->rgpChain[0]->rgpElement[i])  .
    //
    //  ,      (Delta CRL), 
    //  NextIssueDate   NextUpdate  
    //  . 
    //
    PCERT_REVOCATION_INFO pri = NULL;
    PCERT_REVOCATION_CRL_INFO prci = NULL;
    PCCRL_CONTEXT pcc = NULL;
    if(NULL != (pri = pChainContext->rgpChain[0]->
			    rgpElement[0]->pRevocationInfo) &&
       NULL != (prci = pri->pCrlInfo) &&
       NULL != (pcc = ( prci->pDeltaCrlContext 
		    ? prci->pDeltaCrlContext
		    : prci->pBaseCrlContext
		    )
		)
       ) {
	if(CompareFileTime(&pcc->pCrlInfo->NextUpdate, 
			    &ftNotAfter) < 0){
	    ftNotAfter = pcc->pCrlInfo->NextUpdate;
	}
    } else {
        h->LogUfn(h->pUfnArg, CAPI_INFO, 0,
            "Certificate was checked for revocation by OCSP (no CRL info)", policyStatus.dwError);
    }
    ::CertFreeCertificateChain(pChainContext);
#endif /*TEST_CONFORMITY*/

    *ptNotAfter = filetime2time_t(&ftNotAfter);
    *ppCertCtx = pCertContext;
    return CAPI_NOERROR;
}

/*!
 * \ingroup ikeexample
 * \brief      AT_KEYEXCHANGE
 *
 * \xmlonly <locfile><header>ike_cert.h</header></locfile>\endxmlonly
 *
 *       
 *     ike_cert.cpp.
 *
 * \param szProvName [in]  
 * \param szContName [in]  
 * \param szPIN [in] PIN     AT_KEYEXCHANGE  
 * \param phKey [out]   
 * \param phProv [out]   
 *
 * \return CAPI_NOERROR  
 * \return CAPI_CALL_ERROR   
 * \return CAPI_INTERNAL_ERROR    
 *
 */
capi_result
ikeGetPSKKeyFromContainer( const char * szProvName, const char * szContName, const BYTE* szPIN,
			  HCRYPTKEY *phKey, HCRYPTPROV *phProv )
{
    if( !phKey || !phProv )
	return CAPI_CALL_ERROR;

    HCRYPTPROV hProv = 0;
    HCRYPTKEY hKey = 0;
    HCRYPTKEY hVKO = 0;    
    bool isOK = false;
    {
	BYTE bBlob[MAX_PUBLICKEYBLOB_LEN];
	DWORD dwLen = sizeof( bBlob );
	if( !CryptAcquireContextA( &hProv, szContName?szContName:0, szProvName?szProvName:0, PROV_GOST_2001_DH, 0 ) )
	    goto exit;
	if( szPIN && !CryptSetProvParam( hProv, PP_KEYEXCHANGE_PIN, (BYTE *)szPIN, 0 ) )
	    goto exit;
	if( !CryptGetUserKey( hProv, AT_KEYEXCHANGE, &hKey ) ||
	    !CryptExportKey( hKey, 0, PUBLICKEYBLOB, 0, (BYTE *)bBlob, &dwLen ) )
	    goto exit;
	// VKO
	{
	    static BYTE SV[8] = {0xAA,0x40,0x2A,0x8E,0x90,0x1C,0xE3,0xFA};
	    DWORD dwAlg = CALG_G28147;
	    if( !CryptExportKey( hKey, 0, PUBLICKEYBLOB, 0, (BYTE *)bBlob, &dwLen ) ||
		!CryptImportKey( hProv, (BYTE *)bBlob, dwLen, hKey, CRYPT_EXPORTABLE, &hVKO ) ||
		!CryptSetKeyParam( hVKO, KP_SV, SV, 0 ) ||
		!CryptSetKeyParam( hVKO, KP_ALGID, (BYTE *)&dwAlg, 0 ) )
		goto exit;
	}
	isOK = true;
    }
exit:
    if( hVKO && !isOK )
	CryptDestroyKey( hVKO );
    if( hKey )
	CryptDestroyKey( hKey );
    if( hProv && !isOK )
	CryptReleaseContext( hProv, 0 );
    if( isOK )
    {
	*phKey = hVKO;
	*phProv = hProv;
	return CAPI_NOERROR;
    }
    return CAPI_INTERNAL_ERROR;
}

void
ikeReleasePSKKeyFromContainer( HCRYPTKEY hKey, HCRYPTPROV hProv )
{
    CryptDestroyKey( hKey );
    CryptReleaseContext( hProv, 0 );
}
