/* vim:set sw=4 ts=8 fileencoding=cp1251:::WINDOWS-1251[] */
/*
 * Copyright(C) 2004-2013  
 *
 *    , 
 *    .
 *
 *        ,
 * ,    ,
 *     ,
 * ,      
 *     
 *      .
 */

/*!
 * \file $RCSfile$
 * \version $Revision: 271048 $
 * \date $Date:: 2024-04-16 20:38:11 +0300#$
 * \author $Author: raa $
 */

static int log_level = 1;
static int log_format = 0x1f;
static char *serial=(char*)0;

#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,9)
#   error Kernels below 2.6.9 are not supported
#endif

#include <linux/module.h>
#include <linux/vmalloc.h>
#include <linux/init.h>
#ifndef module_param
#   include <linux/moduleparam.h>
#endif
module_param(log_level, int, 0);
module_param(log_format, int, 0);
module_param(serial, charp, 0);
#include <linux/hardirq.h> /* for in_interrupt() */
#include <asm/atomic.h>

typedef int wchar_t;

#include "common.h"

#include "wincspc.h"
#include "wincspc_int.h"

#include "lfmm.h"
#include "lfmm_int.h"
#include "reader/support.h"

#if !defined(__powerpc__) && !defined(__powerpc64__) && !defined(__mips__) && !defined(__aarch64__) && !defined (__arm__) && !defined(__riscv)
#define USE_FAST_CODE
#endif

#ifdef USE_FAST_CODE
#   if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,30)
	    /*   AVX
	     * http://ru.wikipedia.org/wiki/AVX
	     * http://kernelnewbies.org/Linux_2_6_30
	     * http://git.kernel.org/linus/a30469e7921a6dd2067e9e836d7787cfa0105627
	     */
#       define FC_MASK 0
#   else
#       define FC_MASK CSP_FAST_CODE_DISABLE_AVX
#   endif
#   if LINUX_VERSION_CODE >= KERNEL_VERSION(4,2,0)
            /*    4.2 */
#       include <asm/fpu/api.h>
#   elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,28)
	    /* include/asm-x86   
	     * arch/include/asm
	     */
#       include <asm/i387.h>
#   elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,24)
	    /*   asm-i386  asm-x86_64  asm-x86 
	     * http://kernelnewbies.org/Linux_2_6_24
	     */
#       include <asm-x86/i387.h>
#   elif defined(__i386__)
#       include <asm-i386/i387.h>
#   elif defined(__x86_64__) || defined(__AMD64__)
#       include <asm-x86_64/i387.h>
#   else
#       error UNKNOWN PLATFORM
#   endif
#endif  /* USE_FAST_CODE */

#if defined(USE_FAST_CODE) &&						\
    defined(__i386__) && LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
    	//  <asm-i386/i387.h> kernel_fpu_begin(),   
	//   (> 2.5.0),     

	static void inline my_kernel_fpu_begin(void)
	{                                                                       
	    struct thread_info *thread = current_thread_info();                 

	    preempt_disable();                                                  
	    if (thread->status & TS_USEDFPU) {                                  
		__save_init_fpu(thread->task);                                  
		return;                                                         
	    }                                                                   
	    clts();                                                             
	}                                                                       

#   define fc_kernel_fpu_begin() my_kernel_fpu_begin()
#else 
#   define fc_kernel_fpu_begin() kernel_fpu_begin()
#endif

#if defined(USE_FAST_CODE) && LINUX_VERSION_CODE < KERNEL_VERSION(2,6,32)

#   if !defined(X86_CR0_TS)
	    // : asm/processor-flags.h
#       define X86_CR0_TS        0x00000008 /* Task Switched */
#   endif

	static inline BOOL my_interrupted_kernel_fpu_idle(void) {
	    return !(current_thread_info()->status & TS_USEDFPU) &&
		    (read_cr0() & X86_CR0_TS);
	}

    static inline BOOL my_interrupted_user_mode(void) {
        return TRUE;
	//TODO:  2.6.32   ,    
	//  2.6.18:
	//	get_irq_regs() - 
	//	user_mode() - 
	//	user_mode_vm() - 
	//      __get_cpu_var() - 
	//      __irq_regs -  (  set_irq_regs())
        //
        //struct pt_regs *regs = get_irq_regs();
        //return regs && user_mode_vm(regs);
	//
	//static inline struct pt_regs *get_irq_regs(void)
	//{
	//        return __get_cpu_var(__irq_regs);
	//}
    }

    static inline BOOL my_irq_fpu_usable(void) {
        return !in_interrupt() || 
                my_interrupted_user_mode() ||
                my_interrupted_kernel_fpu_idle();
    }

#   define fc_irq_fpu_usable() my_irq_fpu_usable()
#else 
#   define fc_irq_fpu_usable() irq_fpu_usable()
#endif


#define MODNAME "drvcsp"

EXPORT_SYMBOL(CPCGetDefaultConfig);
EXPORT_SYMBOL(CPCCreateProvider);
EXPORT_SYMBOL(CPCInitMemoryLF);

#define STATUS_FILE "status"
/*MODULE_LICENSE("LLC \"Crypto-Pro\"");*/
MODULE_LICENSE("GPL");
MODULE_AUTHOR("CryptoPro Ltd. <support@cryptopro.ru>");
MODULE_DESCRIPTION("CryptoPro CSP kernel module");

#ifdef USE_FAST_CODE
    static CPC_Kernel_Fpu_Begin_Callback kernel_fpu_begin_func;
    static CPC_Kernel_Fpu_End_Callback kernel_fpu_end_func;
#endif

static int
drvcsp_init(void)
{
    printk( KERN_ALERT MODNAME ": Registered\n");
    return 0;
}

static inline void CPCAPI
dprint_func(void *callback_arg, const TCHAR *str)
{
    printk(KERN_INFO "%s\n", str);
}

static inline void CPCAPI
eprint_func(void *callback_arg, const TCHAR *str)
{
    printk(KERN_ERR "%s\n", str);
}

static inline void CPCAPI
elprint_func(void *callback_arg, const TCHAR *str)
{
    printk(KERN_ERR "%s\n", str);
}

static inline void CPCAPI
cprint_func(void *callback_arg, const TCHAR *str)
{
    return dprint_func(callback_arg, str);
}
     
 //      
static unsigned long CPCAPI
get_thread_id(void)
{
    return smp_processor_id();
}

/*  -5  do_gettimeofday(),    API */
static DWORD CPCAPI
gettimeofday_func(CPC_timeval *pTV, LPVOID lpArg)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,0,0)
    struct timespec64 now;
    ktime_get_real_ts64(&now);
    pTV->tv_sec = now.tv_sec;
    pTV->tv_usec = now.tv_nsec/1000;
#else
    struct timeval tv;
    do_gettimeofday (&tv);
    pTV->tv_sec = tv.tv_sec;
    pTV->tv_usec= tv.tv_usec;
#endif
    return S_OK;
}

#ifdef __mips__
static uint32_t mips_count (void)
{
    uint32_t cc;
    __asm__ volatile ("mfc0 %0,$9" :"=r" (cc));
    return cc;
}

static DWORD CPCAPI mips_rdtsc(ULONGLONG *pTSC,void * arg)
{
        *pTSC = mips_count();
        return 0;
}
#endif

#ifdef USE_FAST_CODE
    #if defined(DEBUG) || defined(_DEBUG)
        #define FPU_SUP_STOP SUP_STOP
    #else
    	#define FPU_SUP_STOP 
    #endif
	    
    typedef enum CHECK_BUF_STATUS_ {
        CFB_UNDEFINED = 0x46444e55 /*'UNDF'*/,
        CFB_FREE      = 0x45455246 /*'FREE'*/,
        CFB_FLOATING  = 0x544c4653 /*'SFLT'*/,
    } CHECK_BUF_STATUS;

    typedef struct CHECK_FLOAT_BUF_ {
        CHECK_BUF_STATUS iType;
        size_t	sSize;
    } CHECK_FLOAT_BUF;

    static struct cpksp_fpu_counter {
        atomic_t cnt;
        atomic_t total;
        atomic_t notusable;
        volatile CHECK_FLOAT_BUF *last_success_save;
        volatile CHECK_FLOAT_BUF *last_success_restore;
    } cpkf_fpu_counter = { ATOMIC_INIT(0), ATOMIC_INIT(0), ATOMIC_INIT(0), 
			    0, 0 /*...*/ };


    static BOOL CPCAPI kernel_fpu_begin_func(BYTE *buf, CPC_SIZE_T sz, DWORD bl_len, DWORD op_type )
    {
        CHECK_FLOAT_BUF *ps;

        if(sz < sizeof(*ps)){
            printk(KERN_ERR MODNAME ":%s:%d: Error: sz < sizeof(*ps) "
			"(%p, %d, 0x%x, 0x%x)\n", 
			__FUNCTION__, __LINE__,
			buf, (int)sz, bl_len, op_type);
	    FPU_SUP_STOP;
	    return FALSE;
        }
        ps = (CHECK_FLOAT_BUF *)buf;
        if(ps->iType == CFB_FLOATING){
            printk(KERN_ERR MODNAME ":%s:%d: Error: Assert failed: "
			"ps->iType == KSP_FLOATING\n",
			__FUNCTION__, __LINE__);
	    FPU_SUP_STOP;
	    return FALSE;
        }
        if(ps->iType == CFB_UNDEFINED){
            printk(KERN_ERR MODNAME ":%s:%d: Error: "
			"ps->iType == CFB_UNDEFINED\n",
			__FUNCTION__, __LINE__);
	    FPU_SUP_STOP;
        }

        if(!fc_irq_fpu_usable()) {
	    //    ,     FPU/AVX
	    //  .  ,   FALSE.

	    atomic_inc(&cpkf_fpu_counter.notusable);
	    // :
	    //KdPrint((MODNAME "! KSP_fpu_begin: "
	    //    "KeGetCurrentIrql() return %d > DISPATCH_LEVEL => Can't FPU access\n", 
	    //    ps->kirql));
	    return FALSE;
        }
        ps->sSize = sizeof(*ps);
        ps->iType = CFB_UNDEFINED;

        fc_kernel_fpu_begin();
        ps->iType = CFB_FLOATING;

	atomic_inc(&cpkf_fpu_counter.total);
	atomic_inc(&cpkf_fpu_counter.cnt);
	cpkf_fpu_counter.last_success_save = ps;

        return TRUE;
    }

    static BOOL CPCAPI kernel_fpu_end_func(BYTE * buf, CPC_SIZE_T sz, DWORD op_type)
    {
        CHECK_FLOAT_BUF *ps;

        if(sz < sizeof(*ps)){
            printk(KERN_ERR MODNAME ":%s:%d: Error: sz < sizeof(*ps) "
			"(%p, %d, 0x%x)\n", 
			__FUNCTION__, __LINE__,
			buf, (int)sz, op_type);
	    FPU_SUP_STOP;
	    return FALSE;
        }
        ps = (CHECK_FLOAT_BUF *)buf;
        if(ps->sSize != sizeof(*ps)){
            printk(KERN_ERR MODNAME ":%s:%d: Error: Assert failed: "
			"ps->sSize != sizeof(*ps)\n",
			__FUNCTION__, __LINE__);
            return FALSE;
        }

        switch(ps->iType){
	case CFB_FLOATING:
            kernel_fpu_end();
	    break;
	case CFB_UNDEFINED:
            printk(KERN_ERR MODNAME ":%s:%d: Error: Assert failed: "
			"Not locked (CFB_UNDEFINED)\n",
			__FUNCTION__, __LINE__);
	    FPU_SUP_STOP;
	    return FALSE;
	case CFB_FREE:
            printk(KERN_ERR MODNAME ":%s:%d: Error: Assert failed: "
			"Double free (CFB_FREE)\n",
			__FUNCTION__, __LINE__);
	    FPU_SUP_STOP;
	    ps->iType = CFB_UNDEFINED;
	    return FALSE;
	default:
            printk(KERN_ERR MODNAME ":%s:%d: Error: Assert failed: "
			"Unknown ps->iType\n",
			__FUNCTION__, __LINE__);
	    FPU_SUP_STOP;
	    ps->iType = CFB_UNDEFINED;
	    return FALSE;
        }

	if(0 > atomic_dec_return(&cpkf_fpu_counter.cnt)) {
	    FPU_SUP_STOP;
            printk(KERN_ERR MODNAME ":%s:%d: Error: Assert failed: "
			"0 > cpkf_fpu_counter.cnt\n",
			__FUNCTION__, __LINE__);
	}
	cpkf_fpu_counter.last_success_restore = ps;
	ps->iType = CFB_FREE;
	ps->sSize = 0;

        return TRUE;
    }

#   define KERNEL_BUF_SIZE (16 + sizeof(CHECK_FLOAT_BUF))
#else
#   define KERNEL_BUF_SIZE 16
#endif

DWORD CPCAPI CPCGetDefaultConfig(
    LPCPC_CONFIG pConfig,
    LPVOID pvReserved
    )
{
    if(pConfig->cbSize < sizeof(CPC_CONFIG) || pConfig->cbSize >= 0x4000 ||
       pvReserved != NULL) {
	return (DWORD)E_INVALIDARG;
    }
    memset(
	(char*)&pConfig->FuncStruct, 
	0, 
        pConfig->cbSize - CPRO_OFFSETOF(CPC_CONFIG, FuncStruct));
    pConfig->logConfig.name=MODNAME;
    pConfig->logConfig.level=log_level;
    pConfig->logConfig.format=log_format;
    pConfig->logConfig.dprint_str=dprint_func;
    pConfig->logConfig.eprint_str=eprint_func;
    pConfig->logConfig.elprint_str=elprint_func;
    pConfig->logConfig.cprint_str=cprint_func;
    pConfig->logConfig.get_thread_id=get_thread_id;
    pConfig->timeFuncs.get_time_of_day=gettimeofday_func;
#ifdef __mips__
    pConfig->timeFuncs.read_tsc=mips_rdtsc;
#endif
    #ifdef USE_FAST_CODE
        pConfig->FuncStruct.UsesFunctions=CPC_FAST_CODE_USER;
        pConfig->FuncStruct.cp_kernel_fpu_begin=kernel_fpu_begin_func;
        pConfig->FuncStruct.cp_kernel_fpu_end=kernel_fpu_end_func;
        pConfig->FuncStruct.UsedMask=CSP_OPERATION_ALL|FC_MASK|CSP_FAST_CODE_RT_TS;
    #endif
    pConfig->FuncStruct.Kernel_Buf_Size=KERNEL_BUF_SIZE;
    pConfig->license=0;

    return 0;
}

static void
drvcsp_exit(void)
{
#ifdef USE_FASTCODE    
    printk(KERN_ALERT MODNAME ": FPU stats: cnt=%d total=%d notusable=%d "
                              "last_success_save=%p last_success_restore=%p\n",
		atomic_read(&cpkf_fpu_counter.cnt), 
		atomic_read(&cpkf_fpu_counter.total), 
		atomic_read(&cpkf_fpu_counter.notusable),
		cpkf_fpu_counter.last_success_save, 
		cpkf_fpu_counter.last_success_restore);
#endif		
    printk(KERN_ALERT MODNAME ": Unregistered\n");
}

module_init(drvcsp_init);
module_exit(drvcsp_exit);
