/*
 * Copyright (C) 1997-2003 by Objective Systems, Inc.
 *
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the
 * inclusion of the above copyright notice. This software or any other
 * copies thereof may not be provided or otherwise made available to any
 * other person. No title to and ownership of the software is hereby
 * transferred.
 *
 * The information in this software is subject to change without notice
 * and should not be construed as a commitment by Objective Systems, Inc.
 *
 * PROPRIETARY NOTICE
 *
 * This software is an unpublished work subject to a confidentiality agreement
 * and is protected by copyright and trade secret law.  Unauthorized copying,
 * redistribution or other use of this work is prohibited.
 *
 * The above notice of copyright on this source code product does not indicate
 * any actual or intended publication of such source code.
 *
 *****************************************************************************/
/** 
 * @file rtStream.h 
 * Common runtime constants, data structure definitions, and run-time functions
 * to support the streaming operations.
 */
#ifndef _RTSTREAM_H_
#define _RTSTREAM_H_

#if !defined NO_ASN1_RT_REDUNDANT
#   define NO_ASN1_RT_REDUNDANT
#endif /* !NO_ASN1_RT_REDUNDANT */

#include "asn1type.h"
#include "asn1intl.h"
#include "rtSocket.h"

#ifdef __cplusplus
extern "C" {
#endif

struct OSRTStream;

/** 
 * @defgroup stream Stream structure definitions.
 * @ingroup comruntime 
 * @{
 */
/**
 * Pointer to stream read function. User may implement his own read
 * function for any specific input streams.
 */
typedef int (*OSRTStreamReadProc)(struct OSRTStream* pStream, 
                                  ASN1OCTET* pbuffer, int bufsize);
/**
 * Pointer to stream flush function. User may implement his own read
 * function for any specific output streams.
 */
typedef int (*OSRTStreamFlushProc)(struct OSRTStream* pStream);

/**
 * Pointer to stream close function. User may implement his own read
 * function for any specific streams.
 */
typedef int (*OSRTStreamCloseProc)(struct OSRTStream* pStream);

/**
 * Pointer to stream write function. User may implement his own read
 * function for any specific output streams.
 */
typedef int (*OSRTStreamWriteProc)(struct OSRTStream* pStream, 
   const ASN1OCTET* data, int numocts);

/**
 * Pointer to stream skip function. User may implement his own read
 * function for any specific input streams.
 */
typedef int (*OSRTStreamSkipProc) (struct OSRTStream* pStream, int skipBytes);

#define OSRTSTRMF_INPUT    0x0001
#define OSRTSTRMF_OUTPUT   0x0002
#define OSRTSTRMF_BUFFERED 0x8000

#define OSRTSTRMF_BUF_INPUT    (OSRTSTRMF_INPUT|OSRTSTRMF_BUFFERED)
#define OSRTSTRMF_BUF_OUTPUT   (OSRTSTRMF_OUTPUT|OSRTSTRMF_BUFFERED)

#if !defined NO_ASN1_RT_REDUNDANT
#define OSRTSTRMID_FILE   1
#define OSRTSTRMID_SOCKET 2
#endif	/* !NO_ASN1_RT_REDUNDANT */

#define OSRTSTRMID_MEMORY 3
#define OSRTSTRMID_USER   1000

#define OSRTSTRM_K_BUFSIZE 1024

#define OSRTSTRM_K_INVALIDMARK ((ASN1UINT)-1)

#define OSRTSTREAM_BYTEINDEX(pctxt) \
((pctxt)->pStream->bytesProcessed + (pctxt)->buffer.byteIndex)

/**
 * This structure is used to define a stream control block for keeping track of
 * stream operations. A user may implement his own specific stream operations
 * by defining read, skip, and close functions for input streams and write,
 * flush, and close for output streams.
 */
typedef struct OSRTStream {
   OSRTStreamReadProc  read;   /**< pointer to read function  */
   OSRTStreamWriteProc write;  /**< pointer to write function */
   OSRTStreamFlushProc flush;  /**< pointer to flush function */
   OSRTStreamCloseProc close;  /**< pointer to close function */
   OSRTStreamSkipProc  skip;   /**< pointer to skip function  */
   void*     extra;          /**< pointer to stream-specific data      */

   ASN1UINT  id;             /**< id of stream (see OSRTSTRMID_* macros  */
   ASN1UINT  bufsize;        /**< physical size of pctxt->buffer.data buf */
   ASN1UINT  readAheadLimit; /**< read ahead limit (used by ::rtStreamMark/::rtStreamReset */
   ASN1UINT  bytesProcessed; /**< the number of bytes already processed */
   ASN1UINT  markedBytesProcessed; /**< the marked number of bytes already processed */
   ASN1UINT  ioBytes;        /**< the actual number of bytes already read/written    */

   ASN1USINT flags;          /**< flags (see OSRTSTRMF_* macros          */

} OSRTStream;

/** @} stream */
/** @defgroup rtStream Low-level unbuffered stream functions.
 * @ingroup comruntime 
 * This group of stream functions is used for unbuffered stream I/O. All
 * operations with streams are performed using a context block strucutre. The
 * difference between several kind of streams is only in the opening functions
 * (see ::rtStreamFileOpen, ::rtStreamFileAttach, ::rtStreamSocketAttach,
 * ::rtStreamMemoryCreate, ::rtStreamMemoryAttach). These functions may be used
 * for any input/output operations with streams. Each stream should be
 * initialized first by call to ::rtStreamInit function. After initialization,
 * a stream may be opened for reading or writing by a call to one of the
 * following functions: ::rtStreamFileOpen, ::rtStreamFileAttach,
 * ::rtStreamSocketAttach, ::rtStreamMemoryCreate, ::rtStreamMemoryAttach. @{
 */

EXTERNRT int rtStreamRelease (ASN1CTXT* pctxt);

/** 
 * This function closes the input or output stream and releases any system
 * resources associated with the stream. For output streams this function also
 * flushes all internal buffers to the stream.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamClose (ASN1CTXT* pctxt);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function flushes the output stream and forces any buffered output
 * octets to be written out.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit. Also, the stream must be opened as an
 *                       input stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
EXTERNRT int rtStreamFlush (ASN1CTXT* pctxt);

/**
 * This function initializes a stream part of the context block. This function
 * should be called first before any operation with a stream.
 *
 * @param pctxt        Pointer to context structure variable, for which stream
 *                       to be initialized.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamInit (ASN1CTXT* pctxt);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function reads up to 'bufsize' bytes of data from the input stream into
 * an array of octets. An attempt is made to read as many as bufsize octets,
 * but a smaller number may be read, possibly zero. The number of octets
 * actually read is returned as an integer. This functions blocks until input
 * data is available, end of file is detected, or another error is occurred.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit. Also, the stream must be opened as an
 *                       input stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @param pbuffer      Pointer to a buffer to receive a data.
 * @param bufsize      Size of the buffer.
 * @return             The total number of octets read into the buffer, or
 *                       negative value with error code if any error is
 *                       occurred.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamRead (ASN1CTXT* pctxt, ASN1OCTET* pbuffer, int bufsize);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function skips over and discards the specified amount of data octets
 * from this input stream. The skip method may end up skipping over some
 * smaller number of octets, possibly 0. The actual number of octets skipped is
 * returned.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit. Also, the stream must be opened as an
 *                       input stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @param skipBytes    The number of octets to be skipped.
 * @return             The actual number of bytes skipped, or negative value
 *                       with error code if any error is occurred.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamSkip  (ASN1CTXT* pctxt, int skipBytes);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function writes the specified amount of octets from the specified array
 * to the output stream.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit. Also, the stream must be opened as an
 *                       output stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @param data         The pointer to data to be written.
 * @param numocts      The number of octets to write.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
EXTERNRT int rtStreamWrite (ASN1CTXT* pctxt, const ASN1OCTET* data, int numocts);

/**
 * This function returns the number of processed octets. If the stream was
 * opened as an input stream, then it returns the total number of read octets.
 * If the stream was opened as an output stream, then it returns the total
 * number of written octets. Otherwise, this function returns an error code.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit. Also, the stream must be opened as an
 *                       output stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @return             The total number of processed octets or error code
 *                       (negative value).
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamGetIOBytes (ASN1CTXT* pctxt);
#endif	/* !NO_ASN1_RT_REDUNDANT */
/** @} rtStream */

/** @defgroup rtStreamBuf Buffered stream functions.
 * @ingroup comruntime 
 * The rtStreamBuf* functions are used to operate with buffered streams. They
 * use the 'buffer' field of the ASN1CTXT context block structure as a buffer
 * for input/output operations. These functions are used for performing
 * stream-oriented decoding/encoding. Each buffered stream should be
 * initialized first by a call to ::rtStreamBufInit function. After
 * initialization, the stream may be opened for reading or writing by a call to
 * one of the following functions: ::rtStreamFileOpen,
 * ::rtStreamFileAttach,::rtStreamSocketAttach, ::rtStreamMemoryCreate, or
 * ::rtStreamMemoryAttach. @{
 */

/**
 * This function closes the buffered input or output stream and releases any
 * system resources associated with the stream. For output streams this
 * function also flushes all internal buffers to the stream.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamBufInit.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamBufClose (ASN1CTXT* pctxt);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function flushes the output stream and forces any buffered output
 * octets to be written out.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamBufInit. Also, the stream must be opened as
 *                       an input stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
EXTERNRT int rtStreamBufFlush (ASN1CTXT* pctxt);

/**
 * This function initializes a buffered stream. This function should be called
 * first before any operation with a buffered stream.
 *
 * @param pctxt        Pointer to context structure variable, for which stream
 *                       to be initialized.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamBufInit (ASN1CTXT* pctxt);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * Marks the current position in this input stream. A subsequent call to the
 * ::rtStreamBufReset function repositions this stream at the last marked
 * position so that subsequent reads re-read the same bytes. The readAheadLimit
 * argument tells this input stream to allow that many bytes to be read before
 * the mark position gets invalidated.
 *
 * @param pctxt             Pointer to context structure variable, which is
 *                            initialized for streaming operations by call to
 *                            ::rtStreamBufInit. Also, the stream must be
 *                            opened as an input stream by call to appropriate
 *                            function (see ::rtStreamFileOpen,
 *                            ::rtStreamFileAttach, ::rtStreamSocketAttach,
 *                            ::rtStreamMemoryCreate, ::rtStreamMemoryAttach).
 * @param readAheadLimit    the maximum limit of bytes that can be read before
 *                            the mark position becomes invalid.
 * @return                  Completion status of operation: 0 (ASN_OK) =
 *                            success, negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamBufMark (ASN1CTXT* pctxt, int readAheadLimit);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function pre-reads a 'size' number of octets of data from the input
 * stream into an internal stream buffer. This functions blocks until all
 * necessary data is read, end of file is detected, or another error occurs.
 * This function guarantees that the internal buffer will contain at least the
 * 'size' number of octets. Thus, it is possible to obtain direct access to the
 * memory buffer with pre-read data by using the 'buffer' field of the ASN1CTXT
 * context block structure.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamBufInit. Also, the stream must be opened as
 *                       an input stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @param size         Number of octets to be pre-read.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamBufPreRead (ASN1CTXT* pctxt, int size);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function reads 'size' number of bytes of data from the input stream
 * into an array of octets. If all or some octets already are in the internal
 * buffer, then these octets will be copied from the internal buffer to the
 * specified buffer. In contrast to the ::rtStreamRead function, this function
 * blocks until all requested input data is read, end of file is detected, or
 * another error occurs.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamBufInit. Also, the stream must be opened as
 *                       an input stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @param pdata        Pointer to a buffer to receive a data.
 * @param size         Size of the buffer.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamBufRead (ASN1CTXT* pctxt, ASN1OCTET* pdata, int size);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function skips over and discards the specified amount of data octets
 * from the input stream. The skip method may end up skipping over some smaller
 * number of octets, possibly 0. The actual number of octets skipped is
 * returned.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamBufInit. Also, the stream must be opened as
 *                       an input stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @param skipBytes    The number of octets to be skipped.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamBufSkip  (ASN1CTXT* pctxt, int skipBytes);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function repositions the stream to the position at the time the mark
 * method was last called on the input stream.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamBufInit. Also, the stream must be opened as
 *                       an input stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamBufReset (ASN1CTXT* pctxt);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function writes the specified amount of octets from the specified array
 * to the buffered output stream. Ordinarily this function stores bytes from
 * the given array into the stream buffer, flushing the buffer to the
 * underlying output stream as needed. If the requested length is at least as
 * large as this stream's buffer, however, then this method will flush the
 * buffer and write the bytes directly to the underlying output stream.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamBufInit. Also, the stream must be opened as
 *                       an output stream by call to appropriate function (see
 *                       ::rtStreamFileOpen, ::rtStreamFileAttach,
 *                       ::rtStreamSocketAttach, ::rtStreamMemoryCreate,
 *                       ::rtStreamMemoryAttach).
 * @param data         The pointer to data to be written.
 * @param numocts      The number of octets to write.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
EXTERNRT int rtStreamBufWrite (ASN1CTXT* pctxt, 
                               const ASN1OCTET* data, 
                               int numocts);
/** @} rtStreamBuf */

/** @defgroup rtStreamImp File, socket, and memory streams */
/** 
 * This function attaches the existing file structure pointer to the stream.
 * The file should have already been opened either for reading or writing. The
 * 'flags' parameter specifies the access mode for the stream - input or
 * output.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit or ::rtStreamBufInit.
 * @param pFile        Pointer to FILE structure. File should be already opened
 *                       either for the writing or reading.
 * @param flags        Specifies the access mode for the stream:
 *                       - OSRTSTRMF_INPUT = input (reading) stream;
 *                       - OSRTSTRMF_OUTPUT = output (writing) stream.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamFileAttach (ASN1CTXT* pctxt, 
                                 FILE* pFile, ASN1USINT flags);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function opens a file stream. The 'flags' parameter specifies the
 * access mode for the stream - input or output.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit or ::rtStreamBufInit.
 * @param pFilename    Pointer to null-terminated string that contains the name
 *                       of file.
 * @param flags        Specifies the access mode for the stream:
 *                       - OSRTSTRMF_INPUT = input (reading) stream;
 *                       - OSRTSTRMF_OUTPUT = output (writing) stream.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamFileOpen (ASN1CTXT* pctxt, 
                               const char* pFilename, ASN1USINT flags);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/**
 * This function attaches an existing socket handle to a stream. The socket
 * should be already opened and connected. The 'flags' parameter specifies the
 * access mode for the stream - input or output.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit or ::rtStreamBufInit.
 * @param socket       The socket handle, created by ::rtSocketCreate. The
 *                       connection for this socket should be already opened by
 *                       using ::rtSocketConnect or ::rtSocketAccept functions.
 * @param flags        Specifies the access mode for the stream:
 *                       - OSRTSTRMF_INPUT = input (reading) stream;
 *                       - OSRTSTRMF_OUTPUT = output (writing) stream.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
#if !defined NO_ASN1_RT_REDUNDANT
EXTERNRT int rtStreamSocketAttach (ASN1CTXT* pctxt, 
                                   OSRTSOCKET socket, ASN1USINT flags);
#endif	/* !NO_ASN1_RT_REDUNDANT */

/* memory streams */
/**
 * This function opens a memory stream (i.e. a stream to or from a dynamic byte
 * array in memory). A dynamic memory buffer will be created by this function.
 * The 'flags' parameter specifies the access mode for the stream - input or
 * output.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit or ::rtStreamBufInit.
 * @param flags        Specifies the access mode for the stream:
 *                       - OSRTSTRMF_INPUT = input (reading) stream;
 *                       - OSRTSTRMF_OUTPUT = output (writing) stream.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
EXTERNRT int rtStreamMemoryCreate (ASN1CTXT* pctxt, ASN1USINT flags);

/**
 * This function opens a memory stream by using a specified fixed-size memory
 * buffer. The 'flags' parameter specifies the access mode for the stream -
 * input or output.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for streaming operations by call to
 *                       ::rtStreamInit or ::rtStreamBufInit.
 * @param pMemBuf      The pointer to the buffer.
 * @param bufSize      The size of the buffer.
 * @param flags        Specifies the access mode for the stream:
 *                       - OSRTSTRMF_INPUT = input (reading) stream;
 *                       - OSRTSTRMF_OUTPUT = output (writing) stream.
 * @return             Completion status of operation: 0 (ASN_OK) = success,
 *                       negative return value is error.
 */
EXTERNRT int rtStreamMemoryAttach (ASN1CTXT* pctxt, ASN1OCTET* pMemBuf, 
                                   ASN1UINT bufSize, ASN1USINT flags);

/**
 * This function returns a memory stream's buffer pointer and size.
 *
 * @param pctxt        Pointer to context structure variable, which is
 *                       initialized for memory streaming operations by call to
 *                       ::rtStreamMemoryCreate or ::rtStreamMemoryAttach.
 * @param pSize        The pointer to unsigned integer to receive the size of
 *                       buffer.
 * @return             The pointer to memory buffer.
 */                                   
EXTERNRT ASN1OCTET* rtStreamMemoryGetBuffer (ASN1CTXT* pctxt, ASN1UINT* pSize);

/** @} rtStreamImp */

#ifdef __cplusplus
}
#endif

#endif /* _RTSTREAM_H_ */

