/* vim:set sw=4 ts=8 fileencoding=cp1251:::WINDOWS-1251[] */
/*
 * Copyright(C) 2008-2013  
 *
 *    , 
 *    .
 *
 *         
 *  ,     ,    , 
 *    : 
 *   1.        
 *           ,   
 *           . 
 *   2.        
 *           ,   
 *              / 
 *        ,   . 
 * 
 *        
 * /   "  "  -  , 
 *    , ,    , 
 *        
 *  .    ,     
 * ,      ,     
 *      ,    /  
 *  ,    ,  Ѩ 
 * ,   , ,   
 *  ,      
 *   (,     , 
 *  ,  ,    - 
 *    ,       
 *  ),         
 *     
 *
 *  ,    , 
 *         
 *   .
 *
 *  -   
 *     .
 */

/*!
 * \file $RCSfile$
 * \version $Revision: 245343 $
 * \date $Date:: 2022-08-11 13:10:41 +0300#$
 * \author $Author: dim $
 *
 * \brief      () Linux
 *
 */

//#define USE_STD_MM 
/* USE_STD_MM until we fix lockfree one*/
#if defined(__x86_64__) || defined(__AMD64__) || defined(__powerpc64__) || defined(__aarch64__) || defined(__riscv)
#define DRV_LONG long
#elif defined(__i386__) || defined(__mips__) || defined(__arm__)
#define DRV_LONG int
#else
#error UNKNOWN PLATFORM
#endif
#include <linux/version.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,9)
#    error Kernels below 2.6.9 are not supported
#endif
#include <linux/module.h>
#include <linux/fs.h>

#include <linux/init.h>
#ifndef module_param
#  include <linux/moduleparam.h>
#endif

#include <asm/atomic.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,12,0)
#include <linux/uaccess.h>
#else  /*    2.6.9 */
#include <asm/uaccess.h>
#endif	/* Linux-4.12 */

#include <linux/byteorder/generic.h>
typedef int wchar_t;

#include <linux/vmalloc.h>
#include <linux/rwsem.h>

#if defined(USE_VMALLOC)
#   if LINUX_VERSION_CODE >= KERNEL_VERSION(4,2,0)
#       include <asm/fpu/api.h>
#   elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,28)
#       include <asm/i387.h>
#   elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,24)
#       include <asm-x86/i387.h>
#   elif defined(__i386__)
#       include <asm-i386/i387.h>
#   elif defined(__x86_64__) || defined(__AMD64__)
#       include <asm-x86_64/i387.h>
#   else
#       error UNKNOWN PLATFORM
#   endif
#endif

#include "drtcsp_io.h"
#include "reader/tchar.h"
MODULE_LICENSE("FreeBSD");
MODULE_AUTHOR("Alexey A. Godin CryptoPro Ltd. <dedal@cryptopro.ru>");
MODULE_DESCRIPTION("drtcsp for drvcsp.o");

#define MIN_DEBUG_LEVEL 0
#define MAX_DEBUG_LEVEL 9
		
#define MAXCONTEXTS 100 /* Fill in something more reasonable if you wish */

DRTCSP channels[MAX_MINORS];
#ifdef __mips__
uint32_t get_c0(void)
{
        uint32_t res = 0;
        asm volatile ("rdhwr $2,$2\n move %0,$v0":"=r"(res));
        return res;
}

DWORD CPCAPI mips_rdtsc(ULONGLONG *pTSC,void * arg)
{
        *pTSC=get_c0();
        return 0;
}
#endif
int CPCAPI trans(unsigned char *data, size_t len, int dir, void *arg)
{
  int error;
  char **buf=(char**)arg;

  if(dir==FROM_KERNEL)
    error=copy_to_user(*buf, data, len);
  else
    error=copy_from_user(data, *buf, len);
  if(!error) *buf+=len;

  return error;
}

static PDRTCSP get_drtcsp(struct file *file)
{
  PDRTCSP pdrtcsp;
  long instance=MINOR((long)file->private_data); 
  if (instance<0 || instance>=MAX_MINORS)
    pdrtcsp=NULL;
  else  
    pdrtcsp = &channels[instance];
  if (!pdrtcsp)
    printk(KERN_ALERT MODNAME ": %s(): !pdrtcsp, file=0x%lx\n", __func__,
	   (unsigned long)file);
  return pdrtcsp;
}

#ifdef USE_STD_MM

  #if defined(USE_VMALLOC)
    atomic_t total;
    atomic_t fpuowned;

    #if LINUX_VERSION_CODE < KERNEL_VERSION(3,3,0)
      static inline int fc_user_thread_has_fpu(void) {
          return current_thread_info()->status & TS_USEDFPU;
      }
    #else
      static inline int fc_user_thread_has_fpu(void) {
          return current->thread.has_fpu;
      }
    #endif

    static inline int fc_kernel_fpu_owned(void) {
        return !fc_user_thread_has_fpu() && 
		!(read_cr0() & X86_CR0_TS);
    }
  #endif

  /*!
   * \ingroup MemoryManager
   * \brief  
   *
   *   .
   */
  static DWORD CPCAPI
  stdAllocMemory(LPCPC_MEMORY_ARENA pArena, CPC_SIZE_T dwSize, DWORD dwMemPoolId,
                 DWORD dwThreadId, LPVOID *pRes)
  {
    void *ptr;
  
    UNUSED(dwThreadId);
    UNUSED(dwMemPoolId);
  
    #if !defined(USE_VMALLOC)
      ptr = kmalloc(dwSize, GFP_NOWAIT); //  : !(__GFP_WAIT&flags)
    #else
      {
        int iOwnedFPU = fc_kernel_fpu_owned();

	atomic_inc(&total);

        if(iOwnedFPU) {
	  atomic_inc(&fpuowned);
	    //  SVR4 ABI: http://www.sco.com/developers/devspecs/abi386-4.pdf
            // ## http://web.archive.org/web/20120918204130/http://www.x86-64.org/documentation/abi-0.99.pdf
            // ## http://developer.apple.com/library/mac/#documentation/DeveloperTools/Conceptual/LowLevelABI/140-x86-64_Function_Calling_Conventions/x86_64.html#//apple_ref/doc/uid/TP40005035-SW1
            // 
            //        FPU/MMX/SSE 
            //   ,   .   
            // ,  kernel_fpu_begin(), MXCSR  x87 CE 
            //   .
            // 
          kernel_fpu_end(); // stts()+preempt_enable()
        }
        ptr = vmalloc(dwSize); //      
                               // "" ,  FPU.
        if(iOwnedFPU) {
          kernel_fpu_begin(); //      "", 
                              //   clts()+preempt_disable() ( ),
                              //    .
        }
      }
    #endif	

    if(!ptr){
      printk(KERN_ALERT MODNAME ": %s: %d: Too big arg for kmalloc()/vmalloc() - %lu\n",
		__func__, __LINE__, (unsigned long)dwSize);
      return NTE_NO_MEMORY;
    } 
    //CPCSP-7986
    //memset (ptr, 0, dwSize);
    *pRes = ptr;
  
    return S_OK;
  }

  /*!
   * \ingroup MemoryManager
   * \brief  
   */
  static DWORD CPCAPI
  stdFreeMemory(LPCPC_MEMORY_ARENA pArena, VOID *pMem, DWORD dwMemPoolId)
  {
    UNUSED(dwMemPoolId);
    if(!pMem){
      printk(KERN_ALERT MODNAME ": %s: %d: Bad args\n", __func__, __LINE__);
      return NTE_FAIL;
    }
    #if !defined(USE_VMALLOC)
      kfree(pMem);
    #else
      vfree(pMem);
    #endif
  
    return 0;
  }

  static void CPCAPI
  stdValidateMemory(LPCPC_MEMORY_ARENA pArena)
  {
      UNUSED (pArena);
  }

  static void CPCAPI
  stdStatMemory (LPCPC_MEMORY_ARENA pArena, LPCPC_MEMORY_STATS pStats,
                 DWORD dwMemPoolId)
  {
      UNUSED (pArena);
      UNUSED (pStats);
      UNUSED (dwMemPoolId);
  }

  static void CPCAPI
  stdDoneMemory(LPCPC_MEMORY_ARENA pArena)
  {
  }

  static DWORD
  stdInitMemory(LPCPC_MEMORY_ARENA *pArena, LONG *PoolSizes, DWORD nPools)
  {
    static CPC_MEMORY_ARENA Config;
    memset (&Config, 0, sizeof (Config));
    Config.pValidateMemory = stdValidateMemory;
    Config.pDoneMemory = stdDoneMemory;
    Config.pAllocMemory = stdAllocMemory;
    Config.pFreeMemory = stdFreeMemory;
    Config.pStatMemory = stdStatMemory;
    *pArena = &Config;
    return S_OK;
  }
#else
#define LFMM_RED_ZONE_SIZE (4096)
#define LFMM_RED_ZONE_MAGIC (0xDEADC0DE)
  static void FillRedZone(void *ptr)
  {
	LPDWORD p = (LPDWORD)ptr;
	LPDWORD pe = p + LFMM_RED_ZONE_SIZE/sizeof(DWORD);
	while (p < pe)
	    *p++ = LFMM_RED_ZONE_MAGIC;
  }

  static LPBYTE CheckRedZone(void *ptr)
  {
	LPDWORD p = (LPDWORD)ptr;
	LPDWORD pe = p + LFMM_RED_ZONE_SIZE/sizeof(DWORD);
	
	while (p < pe)
	    if (*p++ != LFMM_RED_ZONE_MAGIC)
		return (LPBYTE)(p-1);
	return NULL;
  }

  static HRESULT lfmmInitMemory(LPCPC_MEMORY_ARENA * pArena, mem_block * blk, DWORD maxContexts)
  {
	CPC_LFMM_CONFIG Config;
	LONG PoolSizes[16];
	HRESULT code;
	DWORD dwBigBufferSize = 2*LFMM_RED_ZONE_SIZE;
	LPBYTE lpBufferAlloced = NULL;
	memset (PoolSizes, 0, sizeof(PoolSizes));
	PoolSizes [MP_WORK] += 10000 * (250 + maxContexts); 
	PoolSizes [MP_BIG] += 10000 * (250 + maxContexts); 
	PoolSizes [MP_PRIME_M] += 1024 * (400 + maxContexts); 
	PoolSizes [MP_SEC_M] = 1024 * (400 + maxContexts); 
	PoolSizes [MP_WORK_M] += 2 * 1024 * (400 + maxContexts);
	memset (&Config,0, sizeof(Config));
        Config.Buffer=NULL; /* may sleep on request */
	Config.Size=0;
	blk->buffer=NULL;
	blk->size=0;
	Config.fSMP=TRUE;
	Config.PoolSizes=PoolSizes;
	Config.nPools=16;
	Config.nCPUs=1;
	code=CPCInitMemoryLF(pArena,&Config);
	if (code != NTE_NO_MEMORY)
	  return NTE_FAIL;
	dwBigBufferSize += Config.Size;
	lpBufferAlloced=vmalloc(dwBigBufferSize);
    
	printk(KERN_ALERT MODNAME ": Allocated: %p %x %x\n", lpBufferAlloced, Config.Size, dwBigBufferSize);
	if(!lpBufferAlloced)
	{
		printk(KERN_ALERT MODNAME ": %s: %d: Too big memory alloc %x\n",
		   __func__, __LINE__, Config.Size);
		blk->buffer=NULL;
		blk->size=0;
		return NTE_NO_MEMORY;
	}
	FillRedZone(lpBufferAlloced);
	Config.Buffer = lpBufferAlloced+LFMM_RED_ZONE_SIZE;
	FillRedZone(lpBufferAlloced+LFMM_RED_ZONE_SIZE+Config.Size);
	code=CPCInitMemoryLF(pArena,&Config);
	blk->buffer=Config.Buffer;
	blk->size=Config.Size;
	if (code != S_OK)
	{
	  vfree(lpBufferAlloced);
	  blk->buffer=NULL;
	  blk->size=0;
	  return code;
	}
	return S_OK;
  }

  static void drtcsp_lfmmDoneMemory(mem_block * blk)
  {
	if (blk->buffer && blk->size)
	{
	    LPBYTE lpBufferAlloced = ((LPBYTE)blk->buffer)-LFMM_RED_ZONE_SIZE;
	    LPBYTE lpRightBound = lpBufferAlloced+LFMM_RED_ZONE_SIZE+blk->size;
	    LPBYTE lpLeftCheck, lpRightCheck;
	    lpLeftCheck = CheckRedZone(lpBufferAlloced);  
	    if (lpLeftCheck)
	    {
		printk(KERN_ALERT MODNAME ":Error: left red zone corrupted at offset %u\n", (DWORD)(lpLeftCheck-lpBufferAlloced));
	    }
	    lpRightCheck = CheckRedZone(lpRightBound);
	    if (lpRightCheck)
	    {
		printk(KERN_ALERT MODNAME ":Error: right red zone corrupted at offset %u\n", (DWORD)(lpRightCheck-lpRightBound));
	    }
	    vfree(lpBufferAlloced);
	}
	blk->buffer=NULL;
	blk->size=0;
  }
#endif

typedef struct dts_lnx_rwlock_ {
    struct rw_semaphore s;
    int write_locked;
} dts_lnx_rwlock_t;

static inline dts_lnx_rwlock_t *
dts_lnx_rwlock_imp(LPCPC_RWLOCK pSection)
{
    return (dts_lnx_rwlock_t *)pSection;
}

static DWORD CPCAPI 
dts_lnx_rwlock_init(LPCPC_RWLOCK pSection, DWORD cbSection, LPVOID lpArg)
{
    char cp_c_version_static_assert[(
    		sizeof(dts_lnx_rwlock_t) <= sizeof(*pSection)
		)?1:-1];
    dts_lnx_rwlock_t *prw = dts_lnx_rwlock_imp(pSection);

    UNUSED(cp_c_version_static_assert);
    UNUSED(lpArg);

    if (cbSection<sizeof(dts_lnx_rwlock_t))
	return NTE_BAD_DATA;

    prw->write_locked = 0;
    init_rwsem(&prw->s);

    return S_OK;
}

static VOID CPCAPI
dts_lnx_rwlock_destroy(LPCPC_RWLOCK pSection)
{
    UNUSED(pSection);
}

static VOID CPCAPI
dts_lnx_rwlock_wrlock(LPCPC_RWLOCK pSection)
{
    dts_lnx_rwlock_t *prw = dts_lnx_rwlock_imp(pSection);

    down_write(&prw->s);
    prw->write_locked = 1;
}

static VOID CPCAPI
dts_lnx_rwlock_rdlock(LPCPC_RWLOCK pSection)
{
    dts_lnx_rwlock_t *prw = dts_lnx_rwlock_imp(pSection);

    down_read(&prw->s);
    #if defined(WARN_ON_ONCE)
	WARN_ON_ONCE(prw->write_locked);
    #endif
}

static VOID CPCAPI
dts_lnx_rwlock_unlock(LPCPC_RWLOCK pSection)
{
    dts_lnx_rwlock_t *prw = dts_lnx_rwlock_imp(pSection);

    if(prw->write_locked) {
	prw->write_locked = 0;
	up_write(&prw->s);
    } else {
	up_read(&prw->s);
    }
}

extern int CPCAPI
prepare_config(LPCPC_CONFIG CSPConfig, mem_block * blk, int use_fastcode, int use_locks)
{
    CSPConfig->cbSize = sizeof(CPC_CONFIG);
    {
	const DWORD res = CPCGetDefaultConfig(CSPConfig,NULL);
	if(res)
	    return (int) res;
    }
#ifdef __mips__
    CSPConfig->timeFuncs.read_tsc=mips_rdtsc;
#endif
    if (!use_fastcode)
    {
	CSPConfig->FuncStruct.UsesFunctions=CPC_FAST_CODE_NO;
	CSPConfig->FuncStruct.cp_kernel_fpu_begin=0;
	CSPConfig->FuncStruct.cp_kernel_fpu_end=0;
	CSPConfig->FuncStruct.UsedMask=0;
    }

#ifdef USE_STD_MM
    UNUSED(blk);
    if(stdInitMemory(&CSPConfig->pArena, NULL, 0)) {
      return NTE_NO_MEMORY;
    }
#else
    if(lfmmInitMemory(&CSPConfig->pArena, blk, MAXCONTEXTS)) {
      return NTE_NO_MEMORY;
    }
#endif    
    CSPConfig->logConfig.name=_TEXT(MODNAME);
    if (use_locks)
    {
	    CSPConfig->lockFuncs.rwlock_init=dts_lnx_rwlock_init;
	    CSPConfig->lockFuncs.rwlock_destroy=dts_lnx_rwlock_destroy;
	    CSPConfig->lockFuncs.rwlock_wrlock=dts_lnx_rwlock_wrlock;
	    CSPConfig->lockFuncs.rwlock_rdlock=dts_lnx_rwlock_rdlock;
	    CSPConfig->lockFuncs.rwlock_unlock=dts_lnx_rwlock_unlock;
    }
    return S_OK;
}

extern int CPCAPI
remove_config(LPCPC_CONFIG CSPConfig, mem_block * blk)
{
  UNUSED(CSPConfig);
#ifdef USE_STD_MM
  UNUSED(blk);
#else
  drtcsp_lfmmDoneMemory(blk);
#endif  
   return S_OK;
}


static int
drtcsp_open(struct inode *inode, struct file *file)
{
  int error;
  long instance;
  PDRTCSP pdrtcsp;
  instance=MINOR(inode->i_rdev);
  if (instance<0 || instance>=MAX_MINORS)
    return -EPERM;
  pdrtcsp=&channels[instance];
  pdrtcsp->size = sizeof(*pdrtcsp);
  pdrtcsp->state |= DRTCSP_ATTACH;
  file->private_data=(void *)instance;

  if(0 != (error=open_hook(pdrtcsp)))
    return -error;
  
  return 0;
}

static int
drtcsp_release( struct inode *inode, struct file *file)
{
  PDRTCSP pdrtcsp;
  int error;

  if(!(pdrtcsp=get_drtcsp(file))) return -ENXIO;
  if((error=close_hook(pdrtcsp))) return -error;
  if((error=detach_hook(pdrtcsp))) return -error;

  return 0;
}

static DRV_LONG
drtcsp_read(struct file *file, char *buf, size_t length, loff_t *off)
{
  PDRTCSP pdrtcsp;
  int resid=length;
  char *tmp=buf;
  
  if(!(pdrtcsp=get_drtcsp(file))) return -ENXIO;
  
  return read_hook(pdrtcsp, &resid, &tmp)?-1:length-resid;
}

static DRV_LONG
drtcsp_write(struct file *file, const char *buf, size_t length, loff_t *off)
{
  PDRTCSP pdrtcsp;
  int resid=length;
  char *tmp=(char *)buf;

  if(!(pdrtcsp=get_drtcsp(file))) return -ENXIO;
  return write_hook(pdrtcsp, &resid, &tmp)?-1:length-resid;
}

/* dim XXX    ,    ,    long.  3.3.1  long */
#if LINUX_VERSION_CODE < KERNEL_VERSION(3,3,0)
#   define IOCTL_TYPE int
#else
#   define IOCTL_TYPE long
#endif	/* < 3.3.x */

static IOCTL_TYPE
drtcsp_ioctl(
	#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36) // TODO: 2.6.11 
	    struct inode *inop,
	#endif
	struct file *file, unsigned int cmd, unsigned long arg)
{
  PDRTCSP pdrtcsp;
  int         error = 0;
  DRTCSP_DRIVER_KEY   tmp;

  if(!(pdrtcsp=get_drtcsp(file))) return -ENXIO;

  if((error=ioctl_hook(pdrtcsp))) return -error;

  switch(cmd){
  case DRTCSP_DRIVER_RNG:
    if(copy_from_user((void*)&tmp, (void *)arg, sizeof (tmp))){
      printk(KERN_ALERT MODNAME "copy_from_user failed");
      error = -EFAULT;
      break;
    }

    if((error=set_driver_rng(pdrtcsp, &tmp))) break;

    if(copy_to_user((void *)arg, (void*)&tmp, sizeof (tmp))){
      printk(KERN_ALERT MODNAME "copy_to_user failed");
      error = -EFAULT;
    }
    break;

  case DRTCSP_SESSION_KEY:
    if(copy_from_user((void*)&tmp, (void *)arg, sizeof (tmp))){
      printk(KERN_ALERT MODNAME "copy_from_user failed");
      error = -EFAULT;
      break;
    }

    if((error=set_session_key(pdrtcsp, &tmp))) break;

    if(copy_to_user((void *)arg, (void*)&tmp, sizeof (tmp))){
      printk(KERN_ALERT MODNAME "copy_to_user failed");
      error = -EFAULT;
    }
    break;

  case DRTCSP_SELFTEST:
    if(copy_from_user((void*)&tmp, (void *)arg, sizeof (tmp))){
      printk(KERN_ALERT MODNAME "copy_from_user failed");
      error = -EFAULT;
      break;
    }

    if((error=kernel_selftest(pdrtcsp, &tmp))) break;

    if(copy_to_user((void *)arg, (void*)&tmp, sizeof (tmp))){
      printk(KERN_ALERT MODNAME "copy_to_user failed");
      error = -EFAULT;
    }
    break;

  default:
    error = -ENOTTY;
    break;
  }
  memset(&tmp, '0', sizeof(tmp));

  return error;
}

static struct file_operations fops = {
  .owner =	THIS_MODULE,
  .open =	drtcsp_open,
  .release =	drtcsp_release,
  .read =	drtcsp_read,
  .write =	drtcsp_write,
  #if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36) // TODO: 2.6.11
    .ioctl
  #else
    .unlocked_ioctl
  #endif
           =	drtcsp_ioctl
};

static int Major;
#ifdef CONFIG_DEVFS_FS
static devfs_handle_t handles[DRTCSP_MAX_CHANNEL];
#endif

static int __init
drtcsp_init(void)
{
  int i;
#ifdef CONFIG_DEVFS_FS
  char nodename[sizeof(MODNAME)+2]; /* /dev/drtcsp0 */
#endif
  for (i=0;i<MAX_MINORS;i++)
  {
    memset(&channels[i],0,sizeof(channels[i]));
    channels[i].size=sizeof(channels[i]);
    channels[i].minor=i;
    channels[i].state=0;
  }

  if (init_hook() != 0)
  {
    printk(KERN_INFO ": %s() init_hook failed", __func__);
    return -EPERM;
  }

#ifdef CONFIG_DEVFS_FS
  for(i=0; i<DRTCSP_MAX_CHANNEL; i++){
    sprintf(nodename, "%s%1d", MODNAME, i);
    handles[i]=devfs_register(NULL, MODNAME, DEVFS_FL_AUTO_DEVNUM, 0, 0,
			      S_IFCHR|S_IRWXUGO, &fops, NULL);
    if(!Major&&handles[i]) devfs_get_maj_min(handles[i], &Major, NULL);
    printk(KERN_INFO "%s() %d %lx", __func__, i, (unsigned long)handles[i]);
  }
#else
  Major=register_chrdev(0, MODNAME, &fops);
  if(Major<0) {
    printk(KERN_ERR"%s() Failed to register: error %d.\n", __func__, Major);
    return Major;
  }
#endif /* CONFIG_DEVFS_FS */
  printk(KERN_INFO"%s() Registered at Major %d.\n", __func__, Major);
    
  return 0;
}

static __exit void
drtcsp_exit(void)
{
  #ifdef CONFIG_DEVFS_FS
    int i;
    for(i=0;i<DRTCSP_MAX_CHANNEL;i++) if(handles[i]) devfs_unregister(handles[i]);
    printk(KERN_INFO"%s() Unregistered.\n", __func__);
  #else
    unregister_chrdev( Major, MODNAME);
    printk(KERN_INFO"%s() Unregistered Major %d.\n", __func__, Major);
  #endif
  #if defined(USE_STD_MM) && defined(USE_VMALLOC)
    printk(KERN_ALERT MODNAME ": vmalloc() stats: total=%d fpuowned=%d\n",
		atomic_read(&total), 
		atomic_read(&fpuowned));
  #endif
  fini_hook();
}

module_init(drtcsp_init);
module_exit(drtcsp_exit);

struct drtcsp_control_data_t {
  char s[10];
};

int debug_level = MIN_DEBUG_LEVEL;
